<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use App\Models\EbankingActivationRequest;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Validation\Rule;


class EbankingActivation extends Controller
{
    private $http_headers;
    private $url;

    public function __construct()
    {
        $this->http_headers = [
            'Authorization' => env('EB_AUTHORIZATION'),
            'Accept' => env('EB_ACCEPT'),
            'api_key' => env('EB_API_KEY'),
            'eBanqr-Platform-TenantID' => env('EB_TENANTID'),
        ];
        $this->url = env('EB_URL');
    }

    public function account_number()
    {
        return view('ebanking_activation.account_number');
    }

    public function verify_account_number(Request $request)
    {
        $request->session()->forget('otp_verified');
        $request->validate([
            'account' => 'required',
        ]);

        $account_id  = $request['account'];

        $response = Http::withHeaders(
            $this->http_headers
        )->get("{$this->url}/savingsaccounts/{$account_id}");

        if ($response->ok()) {

            $data = json_decode($response, true);

            $response2 = Http::withHeaders(
                $this->http_headers
            )->get("{$this->url}/clients/{$data['clientId']}");

            $data2 = json_decode($response2, true);

            $response3 = Http::withHeaders(
                $this->http_headers
            )->get("{$this->url}/clients/{$data['clientId']}/identifiers");

            $data3 = json_decode($response3, true);

            $check_if_account_already_activated = User::where('account',$data['accountNo'])->count();

            if($check_if_account_already_activated > 0)
            {
                return back()->withInput()->with('status', "The account is already activated for Ebanking. Try logging in or resetting password");
            }

            // if(!isset($data['firstname']))
            // {
            //     return back()->withInput()->with('status', "Ebanking not active for group accounts");
            // }

            $to_save = [
                'account' => $data['accountNo'],
                'client_id' => $data['clientId'],
                'name' => ucwords(strtolower($data['clientName'])),
                 'gender' => $data2['gender']['name'][0],
                 'date_of_birth' => date('Y-m-d', strtotime("{$data2['dateOfBirth'][0]}-{$data2['dateOfBirth'][1]}-{$data2['dateOfBirth'][2]}")),
                 'phone_number' => $data2['mobileNo'],
                'national_id_number' => isset($data3[0]['documentKey'])? $data3[0]['documentKey']: NULL,
                'comment' => 'OK'
            ];

            $EbankingActivationRequest = EbankingActivationRequest::create($to_save);

            return redirect()->route("activate_account", $EbankingActivationRequest->id);
        }

        $to_save = [
            'account' => $account_id,
            'comment' => json_encode($response->json())
        ];

        $EbankingActivationRequest = EbankingActivationRequest::create($to_save);

        return back()->withInput()->with('status', "Error: Ensure you have entered the correct account number. Contact Nzuri offices if problem persists");
    }


    public function activate_account(EbankingActivationRequest $EbankingActivationRequest)
    {
        return view('ebanking_activation.activate_account', compact('EbankingActivationRequest'));
    }

    public function create_account(Request $request)
    {

        $request->validate([
            'title' => ['nullable','string','max:255'],
            'name' => ['required','string', 'max:255'],
            'gender' => ['required', Rule::in(['M', 'F']), 'max:5'],
            'date_of_birth' => ['required', 'date', 'before:today'],
            'national_id_number' => ['required','string'],
            'phone' => ['required', 'min:10'],
            'work_address' => ['nullable', 'string', 'max:255'],
            'residential_address' => ['nullable', 'string','max:255'],
            'marital_status' => [Rule::in(['single', 'married', 'widow' , 'widower', 'separated'])],
            'spouse_name' => ['nullable','string', 'max:255'],
            'spouse_phone' => ['nullable','min:10'],
            'nok_name' => ['nullable', 'string', 'max:255'],
            'nok_phone' => ['nullable', 'min:10'],
            'occupation' => ['nullable', 'string', 'max:255'],
            'source_of_income' => ['nullable', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);

        $EbankingActivationRequest = EbankingActivationRequest::findOrFail($request['id']);


        $to_save =[
            'account' => $EbankingActivationRequest->account,
            'title' => $request['title'],
            'name' =>  $request['name'],
            'national_id_number' => $request['national_id_number'],
            'occupation' => $request['occupation'],
            'gender' => $request['gender'],
            'date_of_birth' => $request['date_of_birth'],
            'email' => $request['email'],
            'phone' => $request['phone'],
            'work_address' => $request['work_address'],
            'residential_address' => $request['residential_address'],
            'marital_status' => $request['marital_status'],
            'spouse_name' => $request['spouse_name'],
            'spouse_phone' => $request['spouse_phone'],
            'nok_name' => $request['nok_name'],
            'nok_phone' => $request['nok_phone'],
            'source_of_income' => $request['source_of_income'],
            'password'=> Hash::make($request['password'])
        ];

        User::create($to_save);

        return redirect()->route('login')->with('status','Account successfully activated, you may now login with your email and password');
    }

    public function send_otp(EbankingActivationRequest $EbankingActivationRequest)
    {
        $request = request();
        $request->session()->forget('otp_verified');

        $user = EbankingActivationRequest::findOrFail($EbankingActivationRequest->id);
        if($user->phone_number)
        {
            $otp = mt_rand(100000, 999999);
            $message = "Dear {client_name}, Your OTP is {otp} which expires in 10 minutes";
            $message = str_replace(
                ['{client_name}', '{otp}'],
                [$user->name, $otp],
                $message
            );
            $sms_to_post =[
                'user'=>'Nzuri',
                'password'=>'trust1234',
                'sender'=>'smswindow',
                'message'=>$message,
                'reciever'=>$user->phone_number
            ];

            $sms_response = Http::asForm()
                ->post("http://smswindow.net/webapi.php", $sms_to_post);
            if(json_decode($sms_response) !== 1701)
            {
                return redirect()->route("activate_account", $EbankingActivationRequest->id)->with('otp_message_error', 'An error occurred during sending the OTP please contact nzuri Trust');
            }
            $hashedOtp = Hash::make((string)$otp);
            $otpCreationTime = Carbon::now();
            $to_save = [
                'otp' => $hashedOtp,
                'otp_expiry_time' => $otpCreationTime
            ];
            EbankingActivationRequest::where('id', $EbankingActivationRequest->id)->update($to_save);
            return redirect()->route("activate_account", $EbankingActivationRequest->id)->with('otp_message', 'An sms has been sent to your phone number with a One Time Password(OTP)');
        }

        return redirect()->route("activate_account", $EbankingActivationRequest->id)->with('otp_message_error', 'An error occurred during sending the OTP please contact nzuri Trust');

    }
    public function verify_otp(Request $request)
    {
        $user = EbankingActivationRequest::findOrFail($request['id']);
        $request->validate([
            'otp' => ['required', 'max:6', 'min:6']
        ]);
        $isValid = Hash::check((int)$request['otp'], $user['otp']);
        $otpExpiryTime = Carbon::parse($user['otp_expiry_time'])->addMinutes(10);
        if($isValid && Carbon::now()->lt($otpExpiryTime)){
            $request->session()->put('otp_verified', 'SUCCESS');
            return redirect()->route("activate_account", $request['id']);
        } elseif (Carbon::now()->gte($otpExpiryTime))
        {
            return redirect()->route("activate_account", $request['id'])->with('otp_message_error', 'The OTP has expired. Please generate another one');
        } else {
            return redirect()->route("activate_account", $request['id'])->with('otp_message_error', 'The OTP is invalid. Please enter correct OTP');
        }
    }
}
